globals [
  num-sellers
  monopoly-quantity        ;; the amount that would be supplied by the cartel if it behaved as a monopoly
  monopoly-price           ;; the price expected if monopoly-quantity is supplied 
  monopoly-profit          ;; the maximum profit that the cartel could get if it behaved as a monopolist
  total-supply             ;; sum of the actual quantity supplied by all sellers
  perfect-market-quantity  ;; the amount that would be supplied under perfect competition
  perfect-market-price     ;; the price expected under perfect competition
  individual-quota         ;; the quota for each individual seller based on the cartel agreement
  marginal-cost            ;; the cost of producing one more unit of oil
  avg-price                ;; the average price of oil across all transactions
  min-price                ;; the minimum price available for oil on the market
  low-sellers              ;; the agentset of sellers selling oil for min-price
  color-list-1             ;; used to distribute colors for display of sellers
  color-list-2             ;; used to distribute colors display of sellers
  shape-1?                 ;; used to toggle between shapes for display of sellers   
  age-of-news              ;; age of news item sent to sellers.   used to clear old news    
  time-period              ;; tracks current time period 
]

breeds [
  sellers         ;; every hubnet client is a seller
  buyers          
]


sellers-own [
  user-id                         ;; unique user-id, input by the client when they log in  
  last-sale-profit                ;; how much money the user made last time oil was sold at market
  price-1                         ;; the price of the primary, public offer
  price-2                         ;; the price of the side, private offer
  p1-quantity-offered             ;; amount of oil offered per time period at price-1
  p2-quantity-offered             ;; amount of oil offered per time period at price-2
  p1-quantity-available           ;; amount of oil at price-1 available within a time period as buyers start buying
  p2-quantity-available           ;; amount of oil at price-2available within a time period as buyers start buying
  p1-quantity-sold                ;; amount of oil actually sold at the end of the time period at price-1
  p2-quantity-sold                ;; amount of oil actually sold at the end of the time period at price-1
  prev-p2-quantity-sold           ;; amount of oil sold at price-2 during the previous time period
  profit-needed                   ;; profit needed by each cartel member. varies by seller
  strategy                        ;; set by client.  explained in "Make Offer" procedure
  extra-output                    ;; set by client.  part of "Quota-Plus" strategy. explained in "Make Offer" procedure 
  reduced-output                  ;; set by client.  part of "Quota-Minus" strategy. explained in "Make Offer" procedure 
]

buyers-own [
  quantity-demanded               ;; amount demanded by a buyer at any given moment in a time period
  quantity-bought                 ;; amount bought by a buyer at any given moment in a time period
  seller-bought-from              ;; the seller who sold to the buyer last 
  max-price                       ;; the maximum amount a buyer is willing to pay for their ideal quantity
  already-chosen?                 ;; only used to distribute different levels of max-price across buyers 
]

;; ---------------------------------------------------------
;; STARTUP - auto-setup when the model is opened
;; ---------------------------------------------------------
to startup

  ;; identify hubnet client
  hubnet-set-client-interface "COMPUTER" [ "clients/Oil Cartel client.nlogo" ]
  hubnet-reset
    
  ;; Create buyers
  create-custom-buyers num-buyers [
    setxy (random screen-size-x) (random screen-size-y)
    set color green
  ]
  
  ;; set up colors and shape options for sellers
  initialize-shapes
  
  ;; make Current Profit a plot that can vary by client
  __hubnet-make-plot-narrowcast "Current Profit Plot"
  
end

;; ---------------------------------------------------------
;; RESET - reset everything for a new simulation
;; ---------------------------------------------------------
to reset  
 
  ;; clear out current buyers
  ask buyers [die]
  
  ;; Create new buyers
  create-custom-buyers num-buyers [
    setxy (random screen-size-x) (random screen-size-y)
    set color green
  ]
  
  ;; initialize values
  initial-login
  
  ;; initialize colors and shape options for seller shapes
  initialize-shapes
   
  ;; Reset plots
  set-current-plot "Current Profit Plot"
  ask sellers [ __hubnet-clear-plot user-id ]
  
  set-current-plot "Oil Sold on Market"
  clear-plot
  
  set-current-plot "Average Price of Oil"
  clear-plot
    
end

;; ---------------------------------------------------------
;; INIT-GLOBALS - initialize our global variables
;; ---------------------------------------------------------
to init-globals
  set age-of-news 0
  set avg-price 0
  set min-price 0
  set monopoly-profit ( monopoly-quantity * ( monopoly-price - marginal-cost) ) 
  set time-period 0
end


;; ---------------------------------------------------------
;; INITIALIZE-SHAPES - listen for clients and establish them as
;;                 sellers
;; ---------------------------------------------------------
to initialize-shapes
  ;; set up colors for seller shapes
  set color-list-1      (list ["White" white]  ["Red" (red)]  ["Pink" pink] ["Lime" lime] ["Sky Blue" sky] ["Magenta" magenta]
                              ["Cyan" cyan] ["Turquoise" turquoise]  ["Brown" brown] ["Blue" blue])
  set color-list-2 reverse color-list-1
  set shape-1? true
end


;; ---------------------------------------------------------
;; INITIAL-LOGIN - listen for clients and establish them as
;;                 sellers
;; ---------------------------------------------------------
to initial-login

  ;; Clear all of our buyer, seller, and global variables
  init-globals
  init-seller-vars
  init-buyer-vars

  ;; Listen for logins, etc.
  listen-to-clients
    
  ;; set cartel agreement
  update-cartel-agreement
  
  ;; Calculate market conditions
  compute-market-data
      
  ;;initialize seller information   
  update-client-monitors
  if num-sellers != 0 [ hubnet-broadcast "Current Profit" precision (monopoly-profit / num-sellers) 0]
  
  ;; initialize seller strategies
  hubnet-broadcast "Strategy" "Agreement"
  hubnet-broadcast "extra-output" 0
  hubnet-broadcast "reduced-output" 0        
end


;; ---------------------------------------------------------
;; RUN-MARKET - run the market simulation
;; ---------------------------------------------------------
to run-market

  ;; Make sure there are sellers before running the market
  if num-sellers = 0 [
   show "Can't run market with no sellers.  Please start a hubnet client."
   stop
  ]
     
  ;; get current strategies from sellers 
  listen-to-clients
  
  ;; ask sellers to make offers based on their strategies
  make-offers
    
  ;; ask buyers to identify seller(s) with the lowest price
  find-lowest-offer
  
  ;; ask buyers to purchase oil from sellers
  execute-transactions
      
  ;; calculate market conditions based on buyers and sellers most recent actions
  compute-market-data
  
  ;; ask sellers to update individual seller profit information
  update-sellers
  
  ;; ask buyers to update individual buyers demand satisfaction information
  update-buyers
    
  ;; send updated information to client monitors
  update-client-monitors
  
  ;; do plots
  plot-current-profit
  plot-oil-amounts     
  plot-oil-price
  
  ;; update time-period
  set time-period time-period + 1 
  
end

;; ---------------------------------------------------------
;; MAKE-OFFERS - Sellers make concrete offers to market 
;;               based on the strategy they selected. The
;;               strategies are implemented by offering up to two
;;               simultaneous but seperate offers to the buyers.
;; ---------------------------------------------------------
to make-offers
  ask sellers [
  
    ;; save off needed info from previous offer before creating a new one
    let prev-quantity-offered (p1-quantity-offered + p2-quantity-offered)
  
    ;; "Agreement" Strategy:  Produce and price exactly in accordance with 
    ;; the cartel agreement 
    ifelse strategy = "Agreement"
    [ set price-1 monopoly-price
      set p1-quantity-offered individual-quota
      set price-2 monopoly-price
      set p2-quantity-offered 0
     ]
    ;; "Quota-Plus" Strategy: Consistently produce "extra-output" amt beyond the quota and
    ;; offer it to the mkt for a price a little lower than the official agreement price
    [ ifelse strategy = "Quota-Plus"
      [ set price-1 monopoly-price
        set p1-quantity-offered individual-quota
        set price-2 (monopoly-price - 100)
        set p2-quantity-offered extra-output
       ]

      ;; "Quota-Minus" Strategy:  Consistently produce "reduced-output" amt below the quota 
      ;; in an effort to keep prices up
      [ ifelse strategy = "Quota-Minus"
        [ set price-1 monopoly-price
          set p1-quantity-offered (individual-quota - reduced-output)
          set price-2 monopoly-price
          set p2-quantity-offered 0
        ]
        
        ;; "Flood Market" Strategy:  Saturate the market with low cost oil to punish cheaters.
        [ ifelse strategy = "Flood Market"
          [  set price-1 monopoly-price
             set p1-quantity-offered 0
             set price-2 marginal-cost
             set p2-quantity-offered perfect-market-quantity
             send-news-item user-id + " unleashes reserves as warning"
          ]
          
          ;; "Price > MC" Strategy: Keep producing and offering additional output as long
          ;; as the price you have to offer to sell that unit is still higher than the cost
          ;; to produce it. 
          [ set price-1 monopoly-price
            set p1-quantity-offered 0
            if time-period = 0 [
              set p2-quantity-offered individual-quota 
              set price-2 monopoly-price 
              set prev-quantity-offered p2-quantity-offered
            ]  

            ;; if you didn't sell more this time than last time, undercut your own price
            ;; and try the same amount again. 
            ifelse (p2-quantity-sold <= prev-p2-quantity-sold)
            [ ifelse (price-2 - 10) > marginal-cost
               [ set price-2 (price-2 - 10) ]
               [ set price-2 marginal-cost ]  ;; but don't go as far as pricing below cost
              set p2-quantity-offered prev-quantity-offered
            ]
            ;;if you did sell more that last time, increase production even a little more
            ;; (as long as price > mc)
            [ if price-2  > marginal-cost [  
                set p2-quantity-offered (p2-quantity-offered + 10)  ;; amt offered keeps increasing
               ]
            ]               
           
          ]
       ]
    ]
   ]   
   
   ;; intialize amounts available for sale in the next time period to 
   ;; the total amount you are willing to offer
   set p1-quantity-available p1-quantity-offered
   set p2-quantity-available p2-quantity-offered

  ]
end
 


;; ---------------------------------------------------------
;; COMPUTE-MARKET-DATA - recompute the market
;;                        using the latest info                        
;; ---------------------------------------------------------
to compute-market-data
      
  ;; Check the number of sellers
  set num-sellers count sellers
  
  ;; The actual amount supplied
  set total-supply (sum values-from sellers [p1-quantity-sold] + sum values-from sellers [p2-quantity-sold])
   
  ;; Calcualte the average selling price
  ifelse total-supply != 0 
    [ set avg-price ( (sum values-from sellers with [p1-quantity-sold > 0] [price-1 * p1-quantity-sold] +
                       sum values-from sellers with [p2-quantity-sold > 0] [price-2 * p2-quantity-sold])
                       / total-supply ) 
    ]
    [ set avg-price 0 ]
  
  ;; Calculate hypothetical quantity and price under perfect competition
  ;; Economic theory predicts that point is where price equals marginal cost
  set perfect-market-price marginal-cost
  set perfect-market-quantity filter-zero-or-greater (num-buyers * ideal-quantity - perfect-market-price)
     
end
   
;; ---------------------------------------------------------
;; UPDATE-SELLERS - update seller information
;; ---------------------------------------------------------
to update-sellers
  ask sellers [
  
    ;; figure out how much, if any, extra production there was
    let unused-p1-qty (p1-quantity-offered - p1-quantity-sold) ;; amount produced but not sold at price 1
    let extra-produced filter-zero-or-greater (p2-quantity-offered - unused-p1-qty)
        
    ;; update profit info
    set last-sale-profit int ( (p1-quantity-sold * price-1) + (p2-quantity-sold * price-2)
                           - ((p1-quantity-offered + extra-produced) * marginal-cost) ) 
   ]
end

;; ---------------------------------------------------------
;; UPDATE-BUYERS - update buyer information
;; ---------------------------------------------------------
to update-buyers
  ask buyers [
    ;; update color
    ifelse quantity-bought > 0
      [ set color green ]
      [ ifelse perfect-market-price <= max-price 
        [ set color yellow ]
        [ set color red ]
      ]
  ]
end
      

;; ---------------------------------------------------------
;; LISTEN-TO-CLIENTS - handle connecting clients
;; ---------------------------------------------------------
to listen-to-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-seller hubnet-message-source ]
    [
      ifelse hubnet-exit-message?
      [ remove-seller hubnet-message-source ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end


;; ---------------------------------------------------------
;; EXECUTE-COMMAND - execute the command received
;; ---------------------------------------------------------
to execute-command [command]
    ifelse command = "Strategy"
    [ ask sellers with [user-id = hubnet-message-source]
        [ set strategy hubnet-message ]
          ;set p1-quantity-offered hubnet-message 
          ;set p1-quantity-available p1-quantity-offered]
    ]
    [ ifelse command = "extra-output"
      [ ask sellers with [user-id = hubnet-message-source]
          [ set extra-output hubnet-message ]
            ;set p2-quantity-offered hubnet-message
            ;set p2-quantity-available p2-quantity-offered]
      ]
      [ if command = "reduced-output"
        [ ask sellers with [user-id = hubnet-message-source]
            [ set reduced-output hubnet-message ]
        ]
      ] 
    ]
end

;; ---------------------------------------------------------
;; CREATE-NEW-SELLER - create a new seller when a client
;;                     joins
;; ---------------------------------------------------------
to create-new-seller [ id ]
  let max-jumps 10000
  let color-name ""
  let shape-name ""
  
  create-custom-sellers 1
  [
    set user-id id     ;; remember which client this is
    ifelse shape-1? 
      [ set shape "plant1" 
        set color item 1 item 0 color-list-1
        set color-name item 0 item 0 color-list-1
        set shape-name "Circle" 
        set color-list-1 but-first color-list-1
      ]
      [ set shape "plant2" 
        set color item 1 item 0 color-list-2
        set color-name item 0 item 0 color-list-2
        set shape-name "Square"
        set color-list-2 but-first color-list-2
      ]      
    
    set shape-1? not shape-1? 
    set size 3
        
    ;; locate seller
    setxy (random-float screen-size-x) (random-float screen-size-y)   
    while [ any? sellers in-radius 3 and max-jumps > 0 ] 
          [ rt random 360 
            jump random 100 
            set max-jumps (max-jumps - 1) ]

    init-seller-vars
    set heading 0
    
    ;; send color to client
    hubnet-send user-id "You Are:" "The " + color-name + " " + shape-name 
    
    ;; Announce the arrival of a new seller
    send-news-item "A new seller '" + user-id + "' has arrived."
    
  ]
end

;; ---------------------------------------------------------
;; INIT-SELLER-VARS - initialize the seller's variables
;; ---------------------------------------------------------
to init-seller-vars    ;; seller procedure
  
  ask sellers [
    set last-sale-profit 0
    set p1-quantity-sold 0
    set p2-quantity-sold 0
    set price-1 monopoly-price
    set price-2 price-1
    set marginal-cost 500
    set p1-quantity-offered monopoly-quantity
    set p1-quantity-available p1-quantity-offered
    set p2-quantity-available p2-quantity-offered
    set prev-p2-quantity-sold 0
   
    ;; intialize profit needed by each seller to an equal share of the monopoly profit
    if num-sellers != 0 [ set profit-needed (monopoly-profit / num-sellers) ]
  ]
  
  ;; make some sellers require more than an equal share of the monopoly profit, some less
  if num-sellers != 0 [
    ask random-n-of round (num-sellers / 4) sellers [
      set profit-needed (1.25 * profit-needed) 
    ]
    ask random-n-of round (num-sellers / 4) sellers with [ profit-needed = (monopoly-profit / num-sellers )] [
      set profit-needed (0.25 * profit-needed) 
    ]
      
  ]
           
end

;; ---------------------------------------------------------
;; REMOVE-SELLER - remove a seller from the simulation
;; ---------------------------------------------------------
to remove-seller [ id ]
  ask sellers with [ user-id = id ] [ die ]
end

;; ---------------------------------------------------------
;; INIT-BUYER-VARS - initialize the buyer's variables
;; ---------------------------------------------------------
to init-buyer-vars
  ask buyers [ 
    set quantity-demanded ideal-quantity 
    set quantity-bought 0
    set already-chosen? false
  ]
  
  
  ;; Vary the maximum amount a buyer is willing to pay for oil across buyers
  ;; ( Note:  The distribution below results in a aggregate relationship between
  ;; price and quantity that is linear and has a negative slope.  This simple aggregate
  ;; relationship comes in handy when calculating the cartel agreement and the perfect
  ;; competition amounts. )
  let i 0
  let buyers-remaining buyers with [ not already-chosen? ]
  while [ any? buyers-remaining ] [
    ask random-one-of buyers-remaining [
      set max-price (ideal-quantity * i)
      set already-chosen? true
    ]
    set i (i + 1)
    set buyers-remaining buyers with [ not already-chosen? ]
   ]
end


;; ---------------------------------------------------------
;; FIND-LOWEST-OFFER - 
;; ---------------------------------------------------------
to find-lowest-offer
  let sellers-p2-avail (sellers with [p2-quantity-available > 0])
    
  ifelse any? sellers-p2-avail
   [ set min-price min (values-from sellers-p2-avail [price-2])  ]
   [ set min-price monopoly-price ]
        
  ;; identify the seller(s) offering the lowest price
  set low-sellers sellers with [ (price-1 = min-price and p1-quantity-available > 0) or 
                                 (price-2 = min-price and p2-quantity-available > 0) ]
    
end
  
;; ---------------------------------------------------------
;; EXECUTE-TRANSACTIONS - 
;; ---------------------------------------------------------
to execute-transactions   

   ;; before executing transactions, ask sellers to record how much they sold last time
    ;; and to initialize their quantities this time period
    ask sellers [ 
      set prev-p2-quantity-sold p2-quantity-sold
      set p1-quantity-sold 0
      set p2-quantity-sold 0
    ]
 
    ask buyers [
      without-interruption [          
        set quantity-bought 0  ;; initialize quanity-bought this time period
        
        ifelse min-price <= max-price
          [ set quantity-demanded ideal-quantity ]
          [ set quantity-demanded 0 ]
            
       ;; try to buy amount you demand from sellers
       buy-from-sellers
                            
       ;; if you bought oil this round, move close to your seller
       ifelse quantity-bought > 0 
         [ setxy (value-from seller-bought-from 
                        [random-one-of (list (xcor + random-float 1.5) (xcor - random-float 1.5))])
                 (value-from seller-bought-from 
                        [random-one-of (list (ycor + random-float 1.5) (ycor - random-float 1.5))])
         ]
         [ setxy (random screen-size-x) (random screen-size-y) ]
          
     ] ;; end without-interruption
   ] 
end

to buy-from-sellers
     let amt-just-sold 0
     let lucky-one random-one-of low-sellers ;; if more than one seller offers lowest price, pick one randomly
     let avail-from-seller 0
  ;   let which-offer ""


     ;; find out which offer from lucky-seller is the low one
;     ifelse value-from lucky-one [p1-quantity-available] > 0 and value-from lucky-one [p2-quantity-available] > 0 
 ;       [ ifelse value-from lucky-one [price-1] <= value-from lucky-one [price-2]
  ;        [ set which-offer "P1" ]
   ;       [ set which-offer "P2" ]
     ;   ]
    ;    [ ifelse value-from lucky-one [p1-quantity-available] > 0  and value-from lucky-one [p2-quantity-available] = 0
      ;    [ set which-offer "P1" ]
       ;   [ ifelse value-from lucky-one [p1-quantity-available] = 0  and value-from lucky-one [p2-quantity-available] > 0
        ;    [ set which-offer "P2" ]
         ;   [ set which-offer "" ] 
        ;  ]
      ;  ]
          
     ;; figure out the capacity available from the low seller at the min price
     ;; but need to check to see if official price or the side price is lower first
;     ifelse which-offer = "P1"
      ifelse value-from lucky-one [price-1] = min-price  
        [ set avail-from-seller value-from lucky-one [p1-quantity-available] ]
        [ ifelse value-from lucky-one [price-2] = min-price
          [ set avail-from-seller value-from lucky-one [p2-quantity-available] ]
          [ set avail-from-seller 0 ]
        ]
               
     ;; if the current low seller has enough capacity, buy all you need
     ;; otherwise, buy what you can from him
     ifelse avail-from-seller >= quantity-demanded 
       [ set quantity-bought (quantity-bought + quantity-demanded) ]
       [ set quantity-bought (quantity-bought + avail-from-seller) ]
          
     ;; update info of seller you just bought from
     set amt-just-sold quantity-bought         
     ask lucky-one [
   ;    ifelse which-offer = "P1"
      ifelse value-from lucky-one [price-1] = min-price  
         [ set p1-quantity-available (p1-quantity-available - amt-just-sold)  ;; decrement seller's remaining amt available at p1
           set p1-quantity-sold (p1-quantity-sold + amt-just-sold)   ;; increment seller's amt purchased by buyers at p1
         ]
         ;[ if which-offer = "P2"
         [ if value-from lucky-one [price-2] = min-price 
           [ set p2-quantity-available (p2-quantity-available - amt-just-sold)  ;; decrement seller's remaining amt available at p2
             set p2-quantity-sold (p2-quantity-sold + amt-just-sold)   ;; increment seller's purchased by buyers at p2
           ]
         ]
     ]
         
   ;; update your own info 
   set quantity-demanded (quantity-demanded - quantity-bought)   
   set seller-bought-from lucky-one    
   
   ;; if your demand is still not satisfied, try the next seller (if any)
   if quantity-demanded > 0
   [ ifelse any? sellers with [p1-quantity-available > 0 or p2-quantity-available > 0]
         [ find-lowest-offer                   ;; of the sellers with capacity, find the lowest priced ones
           buy-from-sellers                    ;; try to buy from them
         ]
         [stop]  ;; you've tried all the sellers
   ]  

end

;; ---------------------------------------------------------
;; UPDATE-CLIENT-MONITORS - update the info in the clients' displays
;; ---------------------------------------------------------
to update-client-monitors
  
  ;; Send current cartel agreement and market data to all clients
  hubnet-broadcast "Official Price" monopoly-price
  hubnet-broadcast "My Quota" precision individual-quota 0
  hubnet-broadcast "# Suppliers" num-sellers
 ; hubnet-broadcast "Avg Price of Oil" precision avg-price 0
  
  ;; Send individual seller data
  ask sellers [
    hubnet-send user-id "Current Profit" precision last-sale-profit 0
    hubnet-send user-id "Marginal Cost" marginal-cost 
    hubnet-send user-id "Profit Needed" precision profit-needed 0
    hubnet-send user-id "Qty Sold at Official Price" precision p1-quantity-sold 0
    hubnet-send user-id "Extra Qty Sold" precision p2-quantity-sold 0
    hubnet-send user-id "Actual Qty" precision (p1-quantity-sold + p2-quantity-sold) 0
    ifelse p2-quantity-offered != 0 
      [ hubnet-send user-id "Price of Extra Qty" precision price-2 0 ]
      [ hubnet-send user-id "Price of Extra Qty" 0 ]
 
 ]
  
  ;; Clear old news items
  if age-of-news > 15 [ send-news-item " " ]
  set age-of-news age-of-news + 1
end

;; ---------------------------------------------------------
;; UPDATE-CARTEL-AGREEMENT - set ideal quantity for cartel
;; ---------------------------------------------------------
to update-cartel-agreement
 
  ;; Find profit-maximizing quantity assuming cartel behaves as a unitary monopolist.
  ;; Economic theory prescribes that to maximize profits a firm should produce up to the point where 
  ;; Marginal Revenue (1st derivative of demand a firm faces) equals Marginal Cost (1st derivative of total cost).
  ;; The eqn below comes from setting MR = MC and solving for monopoly-quantity.
  set monopoly-quantity filter-zero-or-greater (num-buyers * ideal-quantity - marginal-cost) / 2
  set monopoly-price filter-zero-or-greater (num-buyers * ideal-quantity - monopoly-quantity)

  if num-sellers != 0 [ set individual-quota int (monopoly-quantity / num-sellers) ]
end



;; ---------------------------------------------------------
;; UTILITY PROCEDURES - useful stuff
;; ---------------------------------------------------------

to-report filter-zero-or-greater [ value ]
  ifelse (value >= 0) 
    [ report value ]
    [ report 0 ]
end

to send-news-item [ msg-text ]
  hubnet-broadcast "Market News" msg-text
  set age-of-news 0
end


;; ---------------------------------------------------------
;; PLOTTING PROCEDURES
;; ---------------------------------------------------------
to plot-oil-amounts
  set-current-plot "Oil Sold On Market"
  
  set-current-plot-pen "Agreement"
  plot monopoly-quantity
  
  set-current-plot-pen "Competitive"
  plot perfect-market-quantity
  
  set-current-plot-pen "Actual"
  plot total-supply
end

to plot-oil-price
  set-current-plot "Average Price of Oil"
  
  set-current-plot-pen "Average"
  plot avg-price
  
  set-current-plot-pen "MC"
  plot marginal-cost
  
  set-current-plot-pen "Agreement"
  plot monopoly-price

end

to plot-current-profit
 set-current-plot "Current Profit Plot"

 ask sellers [
  __hubnet-plot user-id last-sale-profit
]
 end
 
 ; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2004 by Uri Wilensky. Updated 2004. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2004).  NetLogo HubNet OilCartel model.
; http://ccl.northwestern.edu/netlogo/models/HubNetOilCartel .
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2004 Uri Wilensky.  All rights reserved.
; See http://ccl.northwestern.edu/netlogo/models/HubNetOilCartel
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***

@#$#@#$#@
GRAPHICS-WINDOW
497
15
915
454
12
12
16.32
1
10
1
1
1
0

CC-WINDOW
5
758
924
853
Command Center

MONITOR
15
231
78
280
# Sellers
num-sellers
0
1

TEXTBOX
10
113
178
131
Buyer Information:

BUTTON
9
30
106
63
initial-login
initial-login
T
1
T
OBSERVER
NIL
NIL

SLIDER
8
169
180
202
ideal-quantity
ideal-quantity
0
100
30
10
1
NIL

BUTTON
9
67
183
100
run-market
run-market
T
1
T
OBSERVER
T
NIL

PLOT
207
19
481
224
Oil Sold on Market
Time
Quantity
0.0
200.0
0.0
1000.0
true
true
PENS
"Actual" 1.0 0 -16776961 true
"Competitive" 1.0 0 -11352576 true
"Agreement" 1.0 0 -65536 true

BUTTON
109
30
182
63
reset
reset
NIL
1
T
OBSERVER
T
NIL

TEXTBOX
8
10
98
28
Activity:

TEXTBOX
17
293
196
311
Current Cartel Agreement:

PLOT
205
264
484
450
Average Price of Oil
NIL
NIL
0.0
200.0
0.0
100.0
true
true
PENS
"Average" 1.0 0 -16776961 true
"MC" 1.0 0 -11352576 true
"Agreement" 1.0 0 -65536 true

MONITOR
16
367
166
416
Target Total Barrels
monopoly-quantity
3
1

MONITOR
15
316
167
365
Official Price
monopoly-price
3
1

SLIDER
8
132
180
165
num-buyers
num-buyers
30
300
200
10
1
NIL

PLOT
352
624
512
744
Current Profit Plot
NIL
NIL
0.0
500.0
0.0
2.0E7
true
false
PENS
"default" 1.0 0 -16776961 true

BUTTON
16
419
166
452
Update Agreement
update-cartel-agreement
NIL
1
T
OBSERVER
T
NIL

TEXTBOX
14
212
164
230
Cartel Information:

@#$#@#$#@
WHAT IS IT?
-----------
The "Oil Cartel" HubNet model is a collaborative exploration of the economics of a market with imperfect competition. As members of a cartel, participants experience how jointly determined price and quantity decisions can be advantageous to suppliers, harmful to consumers, but also why a cartel is so difficult to sustain. In this version of Oil Cartel, cartel members face differing profit expectations, and set production and pricing strategies in an attempt to meet those expectations.  They respond to each other's behavior by altering their strategies. 

HOW IT WORKS
------------
The demand for oil is determined by the number of buyers (NUM-BUYERS) and how much oil they each desire (IDEAL-QUANTITY).  A buyer follows two rules.  One, they always buy from the seller offering the minimum price.  Two, they only buy if the minimum price available does not exceed their maximum willingness to pay (which varies across the buyers).  

Each HubNet Client is a member of the cartel.  The cartel currently has an agreement in place to limit overall production, resulting in one common official price, and a quota for each member. Each member of this cartel independently decides whether to abide by the agreement or to "cheat" on the agreement and try to boost profits by producing and selling beyond their quota. Furthermore, cartel members face different revenue demands ("Profit-Needed") from their home government, as they come from countries of differing levels of economic prosperity.  More specifically, each cartel member can choose from the following production and pricing strategies, which they can alter depending on the respone of buyers and other cartel members:

1. "Agreement" Strategy:  Always produce and price exactly in accordance with the cartel agreement  

2. "Quota-Plus" Strategy: Produce a little bit more than the quota and offer it to the excess amount to the market for a price a little less than the agreement price.  (The additional amount produced is set by the "extra-output" slider. )

3. "Quota-Minus" Strategy:  Produce a little bit less than the quota and offer it to the market at the agreement price. (The amount less is set by the "reduced-output" slider.) A member might do this in an effort to keep prices up and the cartel together. 
4. "Price > MC" Strategy: Keep producing and offering additional output as long
as the price you have to offer to the buyers in order to sell that unit is still higher than the cost to produce it. 

5.  "Flood Market" Strategy:  Saturate the market with low cost oil to punish or send a warning to cheaters.


HOW TO USE IT
-------------
Set the number of buyers  (NUM-BUYERS) and amount of oil each buyer demands (IDEAL-QUANTITY).  Press the INITIAL-LOGIN forever button and leave pressed until all HubNet Client log in. Ask clients to set their initial strategies, paying attention to the the current agreement, current profit, as well as their own profit needed.  Once all clients have selected their initial strategies, press the RUN-MARKET button.  The overall behavior of the market can be witness on the Graphics Window and Plots on the server, which are described below.  Each individual client's information is displayed on their own monitors, inlcuding a plot of their current profit. 

BUTTONS:
--------
INITIAL-LOGIN - Allows sellers to log in to the activity. This forever button should be turned off before beginning the simulation.

RUN-MARKET - Runs the simulation. Clients will only be selling oil while the market simulation is running. Turning off this forever button will stop all market activity, although it can be resumed from the same point by clicking the button again.

RESET - Resets the simulation to its initial state.

UPDATE-AGREEMENT - Updates the current cartel agreement based on current buyer demand.  Only useful if IDEAL-QUANTITY changes, but the simulation is not reset.  

SLIDERS:
--------
NUM-BUYERS - Total number of buyers in the market.  Changing this value requires a reset.  It cannot be changed during the simulation.

IDEAL-QUANTITY -  The amount of oil demanded by each buyer per time period.


MONITORS:
---------
"# Sellers" - The number of sellers participating in the activity.

"Target Total Barrels" - This is the amount of total production agreed upon by the cartel. It is the total of all oil produced by suppliers that would maximize profits if the cartel behaved as a unitary monopolist, given the current buyer demand and supplier marginal cost. This quantity divided by the number of sellers is each sellers quota.

"Official Price" - This is the price of the current cartel agreement. It is the price of oil that corresponds with supplying "Target Total Barrels" given the current buyer demand and supplier marginal cost.    


PLOTS:
------
"Oil Sold at Market" - Plots the quantities of oil sold at market over time. Also on the graph are the amounts that would be supplied under perfect competition and under a monopoly. Anything less than the amount of perfect competiton and above or equal to the monopoly line is an improvement is better for the cartel than a situation with no collusion.   

"Average Price of Oil" - Plots the average price of oil across all transactions in the current time period.  Also on the graph are plots of Official Price of the cartel agreement and the marginal cost of suppliers.  Anything above marginal cost and equal to or below the agreement line is better for the cartel than a situation with no collusion. 

GRAPHICS WINDOW:
----------------

The graphics window shows all the sellers and buyers.  The sellers are represented by by a colored circle or square. The buyers are the multitude of green, yellow, and red  agents in the window.  Green agents are those buyers who were able to make a purchase n the time period.  They move to the seller from which they last purchased oil. Yellow agents are buyers who did not make a purchase, but they would have if it were priced at the level of a perfectly competitive market.  They represent the inefficiency caused by the cartel behavior.  Red agents are buyers who did not purchase oil, nor would they even in a perfectly competitive market.  Price would have to be even more favorable than the perfectly competitive price for these buyers to purchase. 


CLIENT INFORMATION:
-------------------
"You Are:" - Identifies you on the Graphics Window (e.g., The White Circle)

"Market News" - Information of note.

"Marginal Cost" - The cost of producing one more unit.

"# Suppliers" - The number of sellers who are currently in the market.

"Official Price" - Price from the current cartel agreement. 

"Qty Sold at Official Price" -- Amount of oil sold at the agreeement price.

"Price of Extra Qty" - The price at which you offer the extra oil you produce if you choose not to abide by the agreement.  It is determined by the strategy selected.

"Extra Qty Sold" - Amount of oil purchased by buyers at the "Price of Extra Qty"

"My Quota" - Individual member quota based on cartel agreement.

"Actual Qty" - The total amount sold regardless of price.

"Current Profit" - The profit made by the seller last time oil was sold at market.

"Profit-Needed" - The profit expected from you by your home country.

"Current Profit Plot" - A plot over time of the cartel members profit.

"Strategy" - Drop box used to select one of the strategies mentioned above.

"extra-output" -  Slider used to set the extra amount produced if using the "Quota-Plus" strategy.

"reduced-output - Slider used to set the amount production is reduced from quota if using the "Quota-Minus" strategy.


THINGS TO NOTICE
----------------
The cartel can usually withstand a little bit of cheating, even if all the members indulge. If all members pursue the "P > MC" strategy, the market quickly reaches the perfect competition equilibrium price and quanity.  Even if only one member becomes too agressive, the market price of oil will drop, and the cartel will most likely crumble. 


THINGS TO TRY
-------------
Try changing the values of NUM-BUYERS and IDEAL-QUANTITY? Does cheating by cartel members become more or less noticable?

Can you reach an equilibrium point where everyone is exceeded their Profit Needed? 

The "Quota-Minus" strategy seems like an unattractive one.  Can you find a situation where it is a strategy ever actually worth pursuing?


EXTENDING THE MODEL
-------------------
Here are some suggestions for ways to extend the model:

- Create new strategies!  Current strategies are only a fraction of the possible ways a cartel member might behave over time. 

- Give cartel members the capacity to build up reserves which they can then use strategically. Reserves could be a function of a producers maximum capacity, and would limit a sellers ability to "flood the market."


NETLOGO FEATURES
----------------
Client specific plotting.


RELATED MODELS
--------------
Tragedy of the Commons, Gridlock, Prisoner's Dilemma HubNet


CREDITS AND REFERENCES
----------------------

This activity and associated models and materials was created as part of the projects:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT. These projects gratefully acknowledge the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

Copyright 2004 by Uri Wilensky. All rights reserved. 

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed: 
a) this copyright notice is included. 
b) this model will not be redistributed for profit without permission from the copyright holders. 
Contact the copyright holders for appropriate licenses for redistribution for profit. 

To refer to this model in academic publications, please use: Wilensky, U. (2004). NetLogo HubNet Oil Cartel model. http://ccl.northwestern.edu/netlogo/models/HubNetOilCartel. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL. 

In other publications, please use: Copyright 2004 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetOilCartel for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

plant1
true
15
Circle -1 true true 2 2 295
Rectangle -7566196 true false 90 135 120 240
Rectangle -7566196 true false 120 75 135 240
Rectangle -7566196 true false 135 105 180 240
Rectangle -7566196 true false 195 120 225 240
Rectangle -7566196 true false 180 45 195 240
Polygon -65536 true false 105 60 120 75 135 75 105 60
Polygon -65536 true false 165 30 180 45 195 45 165 30

plant2
true
15
Rectangle -1 true true 15 15 285 285
Rectangle -7566196 true false 90 135 120 240
Rectangle -7566196 true false 120 75 135 240
Rectangle -7566196 true false 135 105 180 240
Rectangle -7566196 true false 195 120 225 240
Rectangle -7566196 true false 180 45 195 240
Polygon -65536 true false 105 60 120 75 135 75 105 60
Polygon -65536 true false 165 30 180 45 195 45 165 30
@#$#@#$#@
NetLogo 2.1beta2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
